pro for_uvmodel,ObsPramsStruct,ModPramsStruct,ModSolStruct,Pop2TRegime,r3D,theta3D,phi3D,thmod,phmod,Br,Bth,Bph,VelR,VelTh,VelPh,StokesUV_I,StokesUV_Q,StokesUV_U,nreinit=nreinit,nowidgmess=nowidgmess,fullsphere_dmin=fullsphere_dmin,fullsphere_dmax=fullsphere_dmax

;
; generates UV forward-modeled Stokes parameters based on physical state input
; STILL BEING TESTED
;
;  REQUIRED INPUTS
;
;  R3D,THETA3D,PHI3D -- plasma position in spherical coordinates
;	R3D  in solar radii
;		passed to FOR_UV_STOKES,
;	Phi3D measured clockwise from Sun-observer line (x axis)
;	Theta3D measured clockwise from plane-of-sky vertical (z axis)
;		used in FOR_FIXANGLE transform into Hanle coordinate
;		BIGTHETA and BIGCHI passed to FOR_UV_STOKES
;	THMOD, PHMOD -- model frame of reference global spherical
;	  --needed for calculating non-isotropic radiation from below
;	points are along a line of sight, or an array of lines of sight
;	  with at least 2 points for each LOS vector
;	IN RADIANS
;
; VELR, VELTH, VELPH - solar wind velocity
;	same dimensions as R3D, THETA3D, PHI3D
;	Magnitude calculated below and passed as VWIND into FOR_UV_STOKES
;	(KM/SEC)
;
; BR, BTH, BPH -- magnetic field vector in spherical coordinates
;	in observer's frame of reference
;	IN GAUSS
;		used in FOR_FIXANGLE transform into Hanle coordinate
;		resulting magnetic field vector THETAB, CHIB
;		which is passed into FOR_UV_STOKES
;		along with magnitude BMAG
;
; OBSPRAMSSTRUCT: 
; 	INST - instrument
;	ULIMB -- limb darkening
;
; MODSOLSTRUCT
; 	DENS -- electron density
; 	TEMP -- electron temperature
; 	POP2TREGIME -- if there are more than one population of density
;
; ADDITIONAL INPUTS CAN BE DEFINED AS PART OF MODSOLSTRUCT
;	(so far this has not been implemented -- edits needed in FOR_KINETIC
;	   and probably make_my_cube.pro)
;  or if not, calculated in FOR_KINETIC
; 	used as input to FOR_UV_STOKES

;  W_PAR -- 1/e Gaussian width for velocity distribution of the coronal ion 
;	parallel to the magnetic field (KM/SEC)
;	either passed in as part of ModSolStruct or calculated in FOR_KINETIC
;	Dimensionality same as R3D
;  ANISO -- parameter describing anisotropy of coronal velocity distribution:
;       1/e Gaussian width for velocity distribution of the coronal ion 
;	perpendicular to the magnetic field W_PERP=W_PAR*ANISO; [1 = isotropic distribution]
;	either passed in as part of ModSolStruct or calculated in FOR_KINETIC
;	Dimensionality same as R3D
;
; INPUTS passed through OBSPRAMSSTUCT.SPECTSTRUCT
;	ion-specific used as input to FOR_UV_STOKES
;
;  W_ANG -- 1/e Gaussian width for velocity distribution 
;       of the chromospheric/transition region ion (Angstrom)
;	converted to KM/S to become W_D and used in Doppler dimming, 
;	also used in getting physical units for intensity
;  CHROMORAD -- chromospheric radiation
;  CHROMOHEIGHT -- height of base of scattering cone where the source emission is located
;       single number, default r=1
;       (may vary with theta,phi in future, but this should *NOT* be done vs
;       scattering point theta, phi, rather cone footpoints, which will be a bit trickier)
;  CHROMO_THMIN, CHROMO_THMAX, CHROMO_PHMIN, CHROMO_PHMAX
;       patch of anisotropic (active region) radiation
;	default unset -- all set to 0
;  CHROMOPATCH -- value to multiply by, default 10
;  BLEND -- line blending factor due to transition
;  EINSTEINA - Einstein coefficient for spontaneous emission (1/SEC)
;  EINSTEINB - Einstein coefficient for absorption/stimulated emission (CM^2/SEC)
;  GJ -  Lande factor
;
; INPUTS passed through OBSPRAMSSTUCT.SPECTSTRUCT
;   	Used by FOR_KINETIC
;
;  COLLKEY - whether or not to turn on collisions for total intensity
;	if negative, will return only collisions on I (wont calculate Stokes)
;
;  ISOTROPIC - whether to do isotropic (1) or anisotropic (0) model
;	If set, will overwrite any ModSolStruct.AnIso
;  WPAR_OPEN, WPAR_CLOSED -- multiplicative factors to establish W_PAR if not set in ModSolStruct
;       multiply W_MIN which depends on temperature, ion maxx
;       If info on open vs closed not available, will assume all is closed
;  keyword ANISO_OPEN -- model choice for ANISO for OVI (see for_kinetic.pro)
;       not used if set in ModSolStruct (UV spectro)
;       If info on open vs closed not available, will assume all is closed
;
;  NANGLEINT -- how many steps to take in solid angle integration
;               of the incoming integration
;
; OUTPUTS StokesUV -- Linear polarization, I, Q, U
;
; Called by FOR_UVCALC
;
; Calls FOR_UV_STOKES
;	FOR_FIXANGLE
;	FOR_KINETIC 
;
; Written by Silvano Fineschi, Sarah Gibson 2013
;	Significant update October 2017; SF, SG also Roberto Susino
;	Also May 2019 
;       Oct 2022 -- changed array ( to [ (with help from Vema Panditi)
;	Nov 2022 -- added EinsteinB etc and updated EinsteinA for 1037
;		TO BE VERIFIED!
;		Also changed 1032 blend to 1 
;	Dec 2022 -- added NeVIII (STILL TESTING)
;		also updated NeVIII collisions file in DEFAULTS 
;		  and moved windows file naming correction up to where it mattered
;		also updated EinsteinB and w_ang for NEVIII
;	Jan 2023
;       restructured so abundance, LWidth (w_ang)
;       chromorad,blend, einsteina, einsteinb, and gj 
;       defaults are defined for UV/EUV here
;        (in SpecPrams not here in FOR_UVMODEL.PRO or through ModSolStruct).
;       restructured so w_ang (LWidth), blend, einsteina, and gj defaults are defined for UV/EUV h
;	Also re-ordered code for simplicity
; 
;      May 2023-- allowed for collemiss=-1 to plot just collisional emission no scattering
;	Sep 2023-- 
;		added chromorad explicitly on disk since it will
;		be part of the intensity observed there.
;		This required passing through the CMER central meridian observer position to define the disk.
;		Also, updted Pop2 to include collisions.
;		Also, used abs(chromorad) allowing negative value
;		 to be passed through -- if that happens, won't be added
;		 to the Iline on disk.
;		**NO this was adding it to every integrand. Removed all this and now it is done
;		   in FOR_INTEGRATE
;		-- kept absollute value on chromorad because now it can be negative
;	** also added collisions to pop2 
;     Oct 2023
;	updated for consistency with chianti usage elsewhere in FORWARD
;		updated h, c, Rsun
;     Dec 2023 -- updated to call new rates file including OVI1037 and NEVIII780
;     Feb 2024 -- added rates file and ionization for MGIX
;     Nov 2024 -- added hooks for limb brightening functions -- RK
;     Dec 2024 -- added chromoheight
;     March 2025 -- fixed bug in Velwind
;     May 2025 -- updated for AR radiation from below
;			requires passing through thmod, phmod
;     June 2025 -- passed through FULLSPHERE_DMIN/DMAX
;     July 2025 -- replaced ion density calculation with call to for_iondens
;     October 2025 -- adjusted for 1/e vs equiv width
;
usewindows=0
if strupcase(!version.os_family) eq 'WINDOWS' then usewindows=1

;
; WIDTH CORONA PARALLEL
;  AND ANISOTROPY defaults
;  set in FOR_SPECDEFAULTS 
; 

SpectStructCheck=ObsPramsStruct.SpecPrams
collkey=SpectStructCheck.CollKey
isotropic=SpectStructCheck.Isotropic
nangleint=SpectStructCheck.NAngleInt
wpar_open=SpectStructCheck.Wpar_Open
wpar_closed=SpectStructCheck.Wpar_Closed
aniso_open=SpectStructCheck.AnIso_Open

;
;  VWIND -- magnitude of solar wind speed (direction assumed to be along field)
;     this will be calculated from VelR,VelTh,Velph 
; 	(KM/SEC)
 
Vwind=sqrt(VelR*VelR+VelTh*VelTh+VelPh*VelPh)

; generate model of kinetic temperature

for_kinetic,r3D,isotropic,wpar_open,wpar_closed,aniso_open,ModSolStruct,ObsPramsStruct,w_par,aniso

; ion specific quantities set in FOR_SPECDEFAULTS
;	or FOR_OBSDEFAULTS
;
w_ang=SpectStructCheck.LWidth/sqrt(!dpi)
;  note LWidth set in FOR_SPECDEFAULTS is equiv. width 
;  and needs to be converted to 1/e width for equations below
chromorad=SpectStructCheck.ChromoRad
chromoheight=SpectStructCheck.ChromoHeight
chromo_thmin=SpectStructCheck.Chromo_ThMin
chromo_thmax=SpectStructCheck.Chromo_ThMax
chromo_phmin=SpectStructCheck.Chromo_PhMin
chromo_phmax=SpectStructCheck.Chromo_PhMax
chromopatch=SpectStructCheck.ChromoPatch
; units of photons cm^-2 s^-1 sr^-1 
blend=SpectStructCheck.Blend
einsteinA=SpectStructCheck.EinsteinA
einsteinB=SpectStructCheck.EinsteinB
GJ=SpectStructCheck.GJ
abundance=SpectStructCheck.abundance
wavelength=ObsPramsStruct.Wavelength_Ang

;
; LINE WIDTH CHROMO
;   (band width over which signal is integrated)
;   w_d = 1/e most probable speed equivalent 
;	 to the chromospheric line-width [km/s]

w_d=(w_ang/wavelength)*2.9979d10

; units CM/SEC
;  convert to KM/SEC
w_d=w_d*1d-5

frequency=2.9979d10/(wavelength*1d-8) ; units of s^-1

; integrate over time *of level transition?* 
; chromorad units are  units of photons cm^-2 s^-1 sr^-1
normchromorad=abs(chromorad)/(w_d*frequency/2.9979d5) ; units of photons cm^-2 sr^-1

; collisional rates (from CHIANTI)
; -although Lya from Roberto Susino is different:
;   "For the Lyman-α, I used an IDL routine that was given to me by Petr Heinzel to numerically
;     compute the collisional rates using the five-level hydrogen model atom with continuum 
;     described in Gouttebroze et al. 1993 (A&AS, 99, 513)."

colldir=concat_dir(GET_ENVIRON('FORWARD'),'DEFAULTS')
if strupcase(ObsPramsStruct.instrument) eq 'OVI1032' then collfile=concat_dir(colldir,'OVI1032_rates.txt')
if strupcase(ObsPramsStruct.instrument) eq 'OVI1037' then collfile=concat_dir(colldir,'OVI1037_rates.txt')
;if strupcase(ObsPramsStruct.instrument) eq 'OVI1032' then collfile=concat_dir(colldir,'OVI1032_rates.txt')
if strupcase(ObsPramsStruct.instrument) eq 'LYA' then collfile=concat_dir(colldir,'LYA_rates.txt')
if strupcase(ObsPramsStruct.instrument) eq 'NEVIII770' then collfile=concat_dir(colldir,'NEVIII770_rates.txt')
if strupcase(ObsPramsStruct.instrument) eq 'NEVIII780' then collfile=concat_dir(colldir,'NEVIII780_rates.txt')
if strupcase(ObsPramsStruct.instrument) eq 'MGIX706' then collfile=concat_dir(colldir,'MGIX706_rates.txt')

if usewindows eq 1 then collfile=str_replace(collfile,'/','\')

readcol,collfile,t,c,/silent
crates=interpol(c,t,alog10(ModSolStruct.Temp)) 
;  units of cm^3 s^-1

;  rotate spherical vector into local vertical Hanle coordinates

for_uvangle,r3D,theta3D,phi3D,Br,Bth,Bph,thetaB,chiB,bigtheta,bigchi
Bmag=sqrt(Br^2+Bth^2+Bph^2)

; need to call FOR_UV_STOKES one point at a time, since
; it loops over solid angle which has limits with height dependence
;
; radiative component only
; collisions only affect Stokes I 
; and are done below

dim=size(r3d)
dim1=dim[1]
if dim[0] ne 1 then dim2=dim[2] else dim2=1

StokesUV_I=r3d*0.d0
StokesUV_Q=r3d*0.d0
StokesUV_U=r3d*0.d0
instrument=ObsPramsStruct.instrument
ulimb=ObsPramsStruct.ulimb

; if collkey is negative, will return collisions only
;  for diagnostics purposes

if collkey ge 0 then begin
 for j = 0,dim2-1 do begin
  for i = 0,dim1-1 do begin
    ruse=r3d[i,j] 
    thuse=theta3d[i,j] 
    phuse=phi3d[i,j] 
    thmoduse=thmod[i,j] 
    phmoduse=phmod[i,j] 
    thbuse=thetaB[i,j] 
    chbuse=chiB[i,j]
    bthuse=bigtheta[i,j] 
    bchuse=bigchi[i,j]
    bmuse=Bmag[i,j] 
    vwuse=Vwind[i,j] 
    wpuse=w_par[i,j] 
    anuse=aniso[i,j] 
    for_uv_stokes,ObsPramsStruct,ModPramsStruct,ruse,thuse,phuse,thmoduse,phmoduse,thbuse,chbuse,bthuse,bchuse,bmuse,vwuse,wpuse,anuse,nangleint,w_d,blend,einsteinA,gJ,chromoheight,chromo_thmin,chromo_thmax,chromo_phmin,chromo_phmax,chromopatch,ulimb,instrument,Iuse,Quse,Uuse,nreinit=nreinit,nowidgmess=nowidgmess,fullsphere_dmin=fullsphere_dmin,fullsphere_dmax=fullsphere_dmax
    StokesUV_I[i,j]=Iuse
    StokesUV_Q[i,j]=Quse
    StokesUV_U[i,j]=Uuse
;    print,i,j
  endfor
 endfor
endif

;
; iondens has in it element abundance and ionization fraction

; rpos is a dummy, not needed because nochromoadd=1
; NOTE NOCHROMOADD IS NOT A GLOBAL KEYWORD
; the global way to turn off chromorad as added to the
; disk is to change the sign of chromorad to negative
; which is easy in the widget but line command you need to
; look in for_specdefaults for the value and call for_drive with that
; with a negative sign.
rpos=r3d*0.
for_iondens,ObsPramsStruct,ModSolStruct,ModPramsStruct,r3D,rpos,iondens,nowidgmess=nowidgmess,nochromoadd=1

em_local=einsteinB*normchromorad*(4.*!dpi)      ; units of photons/s

ionemiss=0.83*iondens*einsteinB*normchromorad  ; units of photons cm^-3 s^-1 sr^-1
; includes density in it
; .83 is proton number density 
; for temperatures above 4.5 logT
;
; this introduces back a timescale from EinsteinB
; set up filling factor
; population 2
; NEED TO take into consideration if population 2
;  a different temperature input to FOR_KINETIC and IONIZATION ABOVE
;  also need to put warnings in -- only good for pop2tregime1 
; second coronal population -- a chromospheric population will not
; be treated correctly.
; TALK TO TERRY
;

fill=1.
fillp2=0.
if tag_exist(ModSolStruct,'FillingFactor') then fill=ModSolStruct.FillingFactor
if Pop2TRegime gt 0 then fillp2=ModSolStruct.Pop2FillingFactor

h=6.6262d-27 ; ergs*s
photon_energy = h*frequency ; ergs
RSun_cm = 6.95700d+10  ;solar radius in cm

; StokesUV_* are unitless
StokesUV_I=StokesUV_I*fill*ionemiss*photon_energy ; units of ergs s^-1 cm^-3 sr^-1
StokesUV_Q=StokesUV_Q*fill*ionemiss*photon_energy
StokesUV_U=StokesUV_U*fill*ionemiss*photon_energy

;Add Pop2 electrons

if Pop2TRegime gt 0 then begin
  StokesUV_I=StokesUV_I+StokesUV_I*ModSolStruct.Pop2Dens*fillp2*ionemiss*photon_energy/ModSolStruct.Dens
  StokesUV_Q=StokesUV_Q+StokesUV_Q*ModSolStruct.Pop2Dens*fillp2*ionemiss*photon_energy/ModSolStruct.Dens
  StokesUV_U=StokesUV_U+StokesUV_U*ModSolStruct.Pop2Dens*fillp2*ionemiss*photon_energy/ModSolStruct.Dens
endif

; these below are for benchmarking
; Equivalent of eq.(19) and eq.(20) from GDZ paper [MNRAS 543, 390-406 (2025)]
em_local_jus_rs=StokesUV_I*em_local/(fill*ionemiss*photon_energy)                        ; units of phot/s
emissivity_jus_rs=StokesUV_I/(ModSolStruct.Dens*ModSolStruct.Dens*photon_energy)         ; units of ph cm+3 s^-1 sr^-1

; here we should add collisions 
collemiss=photon_energy*0.83*iondens*ModSolStruct.Dens*crates*fill/(4.d0*!dpi) ; units of ergs s^-1 cm^-3 sr^-1
if Pop2TRegime gt 0 then begin
 collemiss2=photon_energy*0.83*iondens*ModSolStruct.Pop2Dens*crates*fill/(4.d0*!dpi)/ModSolStruct.Dens ; units of ergs s^-1 cm^-3 sr^-1
 collemiss+=collemiss2
endif

; to add collisions
if collkey gt 0 then StokesUV_I += collemiss ; units of ergs s^-1 cm^-3 sr^-1
; or to test just collisions 
;  (scattering will not have been calculated for I, Q, U)
if collkey lt 0 then StokesUV_I = collemiss 
; if collkey lt 0 then Q, U = 0
;
; if collkey = 0 then collemiss not used
;  	scattering only
;
; for benchmarking/comparison to other codes -- 
;  emissivity with and without resonance scattering (photoexcitation)
;  units photons cm+3 s-1 sr-1

em_local_no_rs=collemiss*4.*!dpi/(photon_energy*0.83*iondens*fill)               ; units of phot/s
em_local_with_rs=em_local_jus_rs+em_local_no_rs

emissivity_no_rs=collemiss/(ModSolStruct.Dens*ModSolStruct.Dens*photon_energy)            ; units of ph cm+3 s^-1 sr^-1
emissivity_with_rs=StokesUV_I/(ModSolStruct.Dens*ModSolStruct.Dens*photon_energy)

;
; Finally, we have to multiply by Rsun in cm, to make it an integrand
; in the right units of ergs cm^-2 s^-1 sr^-1
;

StokesUV_I=StokesUV_I*RSun_cm
StokesUV_Q=StokesUV_Q*RSun_cm
StokesUV_U=StokesUV_U*RSun_cm

end
