function for_calc_eui_response_grid,band,hri=hri,fsi=fsi,SpecPrams=SpecPrams,date=date0,$
	pop2tregime=pop2tregime,nowidgmess=nowidgmess
;+
; Project     : ISSI - Cavities
;
; Name        : for_calc_eui_response_grid
;
; Purpose     : Produce a grid in T and Ne containing the Solar Orbiter/EUI High Resolution Imager 
;						174 A responses
;
; Syntax      : resp=for_calc_euifsi_response_grid(band,/ahead,SpecPrams=SpecPrams,date=date)
;
; Inputs:
;			Band - waveband
;
; Input Keywords:
;				In future might have keywords for HRI and FSI
;				SpecPrams - Structure including
;					Abundance, ioneq, cversion, userspecfile, PITeamResp					
;				date - date of observations
;				pop2tregime - only set during emission calculation for second coronal population
;				HRI - EUI/High Resolution Imager
;				FSI- EUI/Full Sun Imager (default)
;
; Outputs     : OutStr - structure containing response and related quantities. 
;				Also saved to a GENX file
;
; Called by FOR_EUV_XRAY_IMAGER_FLUX
;
; Calls PB0R_SolO 
;
; REQUIRES SECCHI PACKAGE
;
; History     : Draft 1-Sep-2024, T. Kucera
;			
;			   
;+

;only coronal band for wither EUI telescopes is 174 A
default,Band,174

specfiles=for_get_specfile_names(SpecPrams,pop2tregime=pop2tregime,nowidgmess=nowidgmess)
NDen=n_elements(specfiles)

if keyword_set(FSI) +keyword_set(HRI) ne 1 then begin
		message,/info,'Should select EITHER FSI (Full Sun Imager) or HRI (High Resolution Imager). Will assume HRI.'
		HRI=1 & FSI=0
endif

;Calculate a lookup table for EUVI for a given set of spectra at different T and Ne

slash=path_sep()
;need to make sure this is the right directory - should be in FORWARD/OBSERVABLES
respfile=GET_ENVIRON('FORWARD')+slash+'OBSERVABLES'+slash+'SolO_EUI_Response.genx'
 		;response units: 'DN ph^{-1} cm^2 sr'
  		;wavelength units: nm  
restgen,file=respfile,EUI_FSI_resp,EUI_HRI_resp

 if keyword_set(HRI) then begin
	resp=EUI_HRI_resp 
	pix_arcsec=for_detpix_lookup('EUI_HRI')
endif else begin
	resp=EUI_FSI_resp
	pix_arcsec=for_detpix_lookup('EUI_FSI')
endelse

;;;;;;;;;Need to go over this again to make sure I got it right

RSun_arcsec=(pb0r_solo(date0,/arcsec))[2] ;(arcsec/RSun)
RSun_cm=6.957d+10    
;RA2ArcSec=!RADeg*3600.    ;deg/rad*arcsec/deg=arcsec/rad						
arcsec2cm=	RSun_cm/RSun_arcsec   ;cm/RSun * RSun/arcsec =cm/arcsec 
;sr2cmSq = (RA2Arcsec*arcsec2cm)^2  ;(arcsec/rad *cm/arcsec)^2 = cm^2/sr

for i=0,NDen-1 do begin		
							;spectrum units: photons cm+3 sr-1 s-1 Angstroms-1
	 restgen,file=specfiles[i],spec  
	 if i eq 0 then begin
		NTemp=n_elements(spec.LogT)
		lookup=dblarr(NTemp,NDen)
		outstr={response:lookup,LogT:spec.LogT,$
				LogNe:fltarr(NDen),$
				band:174,factor:0,$
				ioneq:spec.ioneq,abund:spec.abund,$
				RUnits:'dn cm+3 s-1', version:systime(),Cal_version:1.0}
		if tag_exist(spec,'chianti_version') gt 0 then outstr=add_tag(outstr,spec.chianti_version,'chianti_version')
	 endif else if NTemp ne n_elements(spec.LogT) then message,'Temperature grid size changing'
				; Lambda is in tenths of an Angstrom, the effective area "area_a" is in cm2.
	area_resp  =resp.resp
				;- Resample the effective area to the wavelength grid of the spectrum:
	area_s = interpol(area_resp,10*resp.wav,spec.lambda) 
					;interpolation results in negative values for wavelengths not in 
					; range of resp.wav so let's fix that
	area_s=area_s>0  
				;- Fold spectrum and eff. area to get the number of detected counts
				; DN ph^{-1} cm^2 sr * photons cm+3 sr-1 s-1 Angstroms-1 = DN  cm^5 s-1 Angstroms-1 :
	response_grid=spec.emissivity*(area_s#replicate(1.,NTemp))  ;response for different temperatures
				;cm+5 s-1 DN pix^-1 /((arcsec/sqrt(pix))*(cm/ arcsec))^2= DN cm+3 s-1
	dn=total(response_grid,1)/(pix_arcsec*arcsec2cm)^2	
	lookup[*,i]=dn
	outstr.logne[i]=alog10(spec.density)
endfor ;end i loop over densities

;Adjust so that they don't hit the floating point limits.
outstr.response=lookup*(1d40)
outstr.factor=-40.

return,outstr

end
 
